﻿#*******************************************************************************
#     FONCTION PRINCIPAL                                                       *
#*******************************************************************************
 
function Install-IECompatibilitySiteList
<#
.SYNOPSIS
	Permet d'obtenir la valeur d'une clé de registre.
.DESCRIPTION
	La fonction reçoit un PSObject complet, représentant une clé de registre et retourne sa valeur.
.EXAMPLE
	Get-registre $monRegistre
.PARAMETER Element
	L'objet PSObject comprenant la clé de registre.
.PARAMETER Arg
	Non utilisé
.NOTES
#>
{
[CmdletBinding()]
param ()
 
    $Error.Clear()
 
    $messageListe = @()
 
    $messageListe += ($Message.Success.START_CONFIGURATION | Expand-Message)
 
    if(Test-ActiveDirectoryPresence)
    {
        $messageListe += ($Message.Warning.TEST_ADPRESENCE | Expand-Message)
    }
    else
    {
 
        $registre_integrationlevel = Get-registre -element @{ cheminCle = "HKLM:\SOFTWARE\Policies\Microsoft\Edge"; nom = "InternetExplorerIntegrationLevel" }
        $messageListe += $registre_integrationlevel.messages
 
        $registre_integrationSitelist = Get-registre -element @{ cheminCle = "HKLM:\SOFTWARE\Policies\Microsoft\Edge"; nom = "InternetExplorerIntegrationSiteList" }
        $messageListe += $registre_integrationlevel.messages
 
        $pushRegistryKey = $false

        # La présence de la clé de registre indique que le poste est deja configuré par un administrateur du poste
        if($registre_integrationlevel.OutputVariable -eq 1)
        {
            $messageListe += ($Message.Success.GET_IS_INTEGRATION_OK | Expand-Message)
 
            $registre_integrationSitelist_output = $registre_integrationSitelist.OutputVariable
 
            if($registre_integrationSitelist_output.EndsWith("Edge_IE_Mode.xml"))
            {
                # Si le poste a dejà recu de la part d'une trousse antérieur le fichier XML local, on donne le GO pour pousser la clé de registre avec le nouvel URL du fichier XML
                $pushRegistryKey = $true
            }
            elseif(![String]::IsNullOrEmpty($registre_integrationSitelist_output))
            {
               $messageListe += ($Message.Success.GET_INTEGRATION_LIST | Expand-Message)
            }
        }
        else
        {
            $pushRegistryKey = $true
        }
        
        if($pushRegistryKey)
        {
            
            $registre_integrationlevel = New-Registre -element @{ cheminCle = "HKLM:\SOFTWARE\Policies\Microsoft\Edge"; nom = "InternetExplorerIntegrationLevel"; valeur = 1; registreType = "DWord" }
            $messageListe += $registre_integrationlevel.messages
 
            $registre_integrationSitelist = New-Registre -element @{ cheminCle = "HKLM:\SOFTWARE\Policies\Microsoft\Edge"; nom = "InternetExplorerIntegrationSiteList"; valeur = "https://externes.ramq.gouv.qc.ca/configurations/edge/ext/UrlIEMode.xml"; registreType = "String" }
            $messageListe += $registre_integrationSitelist.messages
 
            $messageListe += ($Message.Success.CONFIG_OK | Expand-Message)
            
        }
    }
 
 
    foreach($msg in $messageListe)
    {
        switch ($msg.Type) 
        {
            "Error"   { Write-Error $msg.Message; break } 
            "Success" { Write-Output $msg.Message; break } 
            "Warning" { Write-Warning $msg.Message; break } 
            default   { Write-Output $msg.Message; break } 
        }
    }
 
}
 
 
 
#*******************************************************************************
#     LISTE DES MESSAGES                                                       *
#*******************************************************************************
$Message = $()
$Message += @{
 
    Error = @{
 
                OPERATIONNONAUTORISEE        = @{ Type="Error"; Message = 'Opération ''$CmdletName'' non-autorisée sur ce serveur.'} 
                TEST_ABSENTE                 = @{ Type="Error"; Message = 'La clé  de registre [$registre_cle] est absente'} 
                RESOLVE_ERROR                = @{ Type="Error"; Message = 'Aucune valeur n''a pu être déterminée pour la clé de registre ''$elementID'' définie dans le fichier des éléments.'}
                COPIE_FICHIERSOURCEMANQUANT  = @{ Type="Error"; Message = 'Le fichier source ''$FicSource'' n''existe pas. Impossible d''effectuer la copie.'}
            }
    Warning = @{
                GET_NONEXISTANT       = @{ Type="Warning"; Message = 'La clé  de registre $registre_cle est absente, impossible d''obtenir sa valeur.'} 
                TEST_VIDE             = @{ Type="Warning"; Message = 'Pour la clé [$registre_cle] la valeur du champ est vide'}
                TEST_ADPRESENCE       = @{ Type="Warning"; Message = 'Ce poste est relié à un Active Directory, aucune configuration ne sera appliqué sur ce poste.'}
            }
    Success = @{
                CONFIG_OK             = @{ Type="Success"; Message = 'SUCCÈS: Votre poste est maintenant prêt à utilisé Edge en mode compatibilité.'}
                COPIE_OK              = @{ Type="Success"; Message = '$FicSource copié vers $FicDest'}
                NEW_CLECREE           = @{ Type="Success"; Message = 'La clé de registre [$registre_cle] est créée'}
                NEW_ASSIGNEVALEUR     = @{ Type="Success"; Message = 'Nouvelle valeur insérée.'}
                NEW_MISEAJOUR         = @{ Type="Success"; Message = 'Valeur mise à jour.'}
                REMOVE_COMPLETE       = @{ Type="Success"; Message = 'La branche de la clé de registre complete [$registre_cle] est détruite'}
                REMOVE_CHAMP          = @{ Type="Success"; Message = 'Le champ [$registre_champ] de la clé de registre complete [$registre_cle] est détruit'}	
                RESOLVE_OK            = @{ Type="Success"; Message = 'La clé de registre ''$elementID'' a correctement été résolu.'}
                TEST_OK               = @{ Type="Success"; Message = '$strInfo'}
                GET_IS_INTEGRATION_OK = @{ Type="Success"; Message = 'Ce poste est déjà configuré pour fonctionner sous Edge en mode compatibilité IE11.'}
                GET_INTEGRATION_LIST  = @{ Type="Success"; Message = 'Ce poste possède une liste de sites fonctionnant en mode compatibilité: $registre_integrationSitelist_output.'}
                START_CONFIGURATION   = @{ Type="Success"; Message = 'Configuration de la liste des sites fonctionnant sous Edge en mode compatibilité Internet Explorer 11...'}
            }       
 
}
 
 
 
#*******************************************************************************
#     FONCTIONS UTILITAIRES                                                    *
#*******************************************************************************
 
 
Function Expand-Message {
    [CmdletBinding()]
    param(
        [parameter(ValueFromPipeline)]
        [Hashtable] $messageObject
    )
    Process {
        $expandedMessage = $ExecutionContext.InvokeCommand.ExpandString($messageObject.Message)
        Write-Output @{ Type=$messageObject.Type; Message=$expandedMessage}
    }
 
}
 
Function Get-LastError
<#
.SYNOPSIS
    Permet de récupérer la dernière erreur.
.DESCRIPTION
    Voir ci-dessus
.EXAMPLE
    Get-LastError
.NOTES
    ..
#>
{
param ()
 
    # l'objet $Error est réinitialisé complétement 
 
    $descriptionErreur = "Une erreur est survenue durant l'opération.`nDESCRIPTION DE L'ERREUR: "
 
    if ($Error.count -ne 0 ) 
    {
        if ($Error[0].Exception -ne $null) 
        {
            $descriptionErreur+=$Error[0].Exception 
        }
        elseif ($Error[0].ErrorRecord -ne $null) 
        {
            $descriptionErreur+=$Error[0].ErrorRecord.Exception
        }
        elseif ($Error[0].Message -ne $null) 
        {
            $descriptionErreur+=$Error[0].Message
        }
        else
        {
            $descriptionErreur+=$Error[0]
        }
    }
 
    return $descriptionErreur
}
 

 
Function Resolve-Message
<#
.SYNOPSIS
    Permet de récupérer la dernière erreur.
.DESCRIPTION
    Voir ci-dessus
.EXAMPLE
    Resolve-Message -Liste $messageListe -Source $CmdletName
.NOTES
    ..
#>
{
param (
    [Parameter(Mandatory = $true)]
    $Liste,
    [Parameter(Mandatory = $true)]
    [string]$Source,
    [Parameter(Mandatory = $false)]
    [PSObject]$OutputVariable
)
 
    $output = New-Object -TypeName PSObject -Property @{
 
                messages = @()
                outputVariable = $OutputVariable
 
            }
 
 
    if ( $Liste ) {
        foreach($message in $Liste)
        {
            if ( $message -ne $null ) {
                $message.sourceMachine = $env:ComputerName
                $message.source = $Source
                $output.messages += $message
            }
        }
    }
 
    if($Error.count -ne 0)
    {
        $output.messages += New-Object -TypeName PSObject -Property @{ 
            Type="Error"
            Message = $(Get-LastError)
            Source = $Source
            sourceMachine = $env:ComputerName
        }
        $Error.Clear()
    }
 
    Write-Output $output
 
}
 
 
Function Test-ActiveDirectoryPresence
<#
.SYNOPSIS
    Permet de vérifier simplement si le poste sur lequel on exécute ce script est relié à unActive Directory.
.DESCRIPTION
    Aucun paramètre d'entré n'est requis
.EXAMPLE
    Test-ActiveDirectoryPresence
.PARAMETER 
    Non-applicable
.RETOUR
    Retour true/false
.NOTES
#>
{
[CmdletBinding()]
param ()
 
    $IsADPresent = $false
    if($env:USERDNSDOMAIN)
    {
        $IsADPresent = $true
    }    
 
    Write-Output $IsADPresent
 
}
 
 
function Get-Registre
<#
.SYNOPSIS
	Permet d'obtenir la valeur d'une clé de registre.
.DESCRIPTION
	La fonction reçoit un PSObject complet, représentant une clé de registre et retourne sa valeur.
.EXAMPLE
	Get-registre $monRegistre
.PARAMETER Element
	L'objet PSObject comprenant la clé de registre.
.PARAMETER Arg
	Non utilisé
.NOTES
#>
{
[CmdletBinding()]
param ([PSObject] $element, [string] $arg)
 
 
	## Recuperer le nom de cette fonction
	[string]$CmdletName = $PSCmdlet.MyInvocation.MyCommand.Name
 
	$regKeyValue = ""
	$messageListe = @()
 
	$registre_champ = $element.nom
	$registre_cle = $element.cheminCle
 
	try 
	{
 
		## Check if the registry key exists
		if (-not (Test-Path -Path $registre_cle -ErrorAction 'Stop')) 
		{
			$messageListe += ($Message.WARNING.GET_NONEXISTANT | Expand-Message)
		}
		else
        {
            if (-not $registre_champ) 
		    {
			    #  Get the Key
			    $regKeyValue = Get-ItemProperty -Path $registre_cle -ErrorAction 'Stop'
		    }
		    else 
		    {
			    #  Get the Value (do not make a strongly typed variable because it depends entirely on what kind of value is being read)
			    $regKeyValue = Get-ItemProperty -Path $registre_cle -ErrorAction 'Stop' | Select-Object -ExpandProperty $registre_champ
		    }
        }	
 
	}
	catch [Exception]
	{
		$messageListe += (@{ Type="Error"; Message = $_.Exception} | Expand-Message)
	}
 
 
 
    Resolve-Message -Liste $messageListe -Source $CmdletName -OutputVariable $regKeyValue
 
 
}
 
 
 
Function New-registre
<#
.SYNOPSIS
	Permet de creer une clé de registre et ou avec un champ.
.DESCRIPTION
	La fonction reçoit un PSObject complet, représentant une clé de registre et vérifie ses caractéristiques.
.EXAMPLE
	Remove-registre $monRegistre
.PARAMETER Element
	L'objet PSObject comprenant la clé de registre.
.PARAMETER Arg
	Non utilisé
.NOTES
#>
{
[CmdletBinding()]
param ([PSObject] $element, [string] $arg)
 
 
	## Recuperer le nom de cette fonction utilisé pour les log d erreurs
	[string]$CmdletName = $PSCmdlet.MyInvocation.MyCommand.Name
 
	[boolean]$blnRetour = $false
	$messageListe = @()
 
	$registre_champ = $element.nom
	$registre_cle = $element.cheminCle
	$registre_valeur = $element.valeur
    $registre_type = $element.registreType
 
 
	try 
	{
 
		if( -not (Test-Path -Path $registre_cle -PathType Container) )
		{
			New-Item -Path $registre_cle -ItemType Registry -Force -ErrorAction 'Stop' | Out-Null
			$messageListe+= ($Message.SUCCESS.NEW_CLECREE | Expand-Message)
		}
 
		## Si on a fourni un champ pour la clé alors on tente de le créer
		if ($registre_champ)
		{
			$properties = Get-ItemProperty -Path $registre_cle 
			if( -not $properties )
			{
				New-ItemProperty -Path $registre_cle -Name $registre_champ -Value $registre_valeur -PropertyType $registre_type -EA Stop -Force | Out-Null
				$messageListe+= ($Message.SUCCESS.NEW_ASSIGNEVALEUR | Expand-Message)
			}
 
			if($properties)
			{
				## mettre a jour la valeur si le champ existe
				Set-ItemProperty -Path $registre_cle -Name $registre_champ -Value $registre_valeur -EA Stop -Force | Out-Null
				$messageListe+= ($Message.SUCCESS.NEW_MISEAJOUR | Expand-Message)               
			}
 
            $blnRetour = $true
		}
 
 
	}
	catch [Exception]
	{
		$messageListe += (@{ Type="Error"; Message = $_.Exception} | Expand-Message)
	}
 
	Resolve-Message -Liste $messageListe -Source $CmdletName -OutputVariable $blnRetour
 
}
 
#$script_root = (split-path -parent ($MyInvocation.MyCommand).Path)
 
#Install-IECompatibilitySiteList
 
  
#New-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Edge" -Name "InternetExplorerIntegrationLevel" -Value "1" -PropertyType DWord -Force
#New-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Edge" -Name "InternetExplorerIntegrationSiteList" -Value "D:\Temp\Edge_IE_Mode.xml" -PropertyType String -Force